<?php
session_start();

/* ==========================================================
   BASIS & CONFIG
   ========================================================== */
$basis = rtrim(dirname($_SERVER['SCRIPT_NAME']), '/\\') . '/';

if (file_exists(__DIR__ . '/config.php')) {
    require_once __DIR__ . '/config.php';
} else {
    define('ADMIN_EMAIL', '');
    define('EMAIL_ENABLED', false);
    define('SALT', 'default_salt_change_me');
}

/* ==========================================================
   ABBRUCH MIT MELDUNG (UX-SICHER)
   ========================================================== */
function abortKommentar(string $msg, string $returnUrl) {
    $_SESSION['kommentar_error'] = $msg;
    header('Location: ' . $returnUrl);
    exit;
}

/* ==========================================================
   NUR POST
   ========================================================== */
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    die('Ungültige Anfrage');
}

/* ==========================================================
   GRUNDWERTE FRÜH
   ========================================================== */
$ziel    = trim($_POST['ziel'] ?? 'allgemein');
$version = trim($_POST['version'] ?? '');

$returnUrl = $basis . 'kommentar.php?ziel=' . rawurlencode($ziel)
           . ($version ? '&version=' . rawurlencode($version) : '');

/* ==========================================================
   DB
   ========================================================== */
try {
    $db = new PDO("sqlite:" . __DIR__ . "/db.sqlite");
    $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (Exception $e) {
    abortKommentar('Datenbankfehler.', $returnUrl);
}

/* ==========================================================
   1. CSRF
   ========================================================== */
if (
    empty($_POST['csrf_token']) ||
    empty($_SESSION['csrf_token']) ||
    !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])
) {
    abortKommentar('Session abgelaufen. Bitte Seite neu laden.', $returnUrl);
}

/* ==========================================================
   2. HONEYPOT (STILL)
   ========================================================== */
foreach (['website','email','url'] as $hp) {
    if (!empty($_POST[$hp])) {
        header('Location: ' . $returnUrl);
        exit;
    }
}

/* ==========================================================
   3. ZEITCHECK
   ========================================================== */
$formTime = (int)($_POST['form_time'] ?? 0);
if ($formTime && (time() - $formTime) < MIN_SUBMIT_TIME) {
    abortKommentar('Bitte Formular langsamer ausfüllen.', $returnUrl);
}
if ($formTime && (time() - $formTime) > MAX_FORM_AGE) {
    abortKommentar('Formular abgelaufen.', $returnUrl);
}

/* ==========================================================
   4. INPUT & BASIS-VALIDIERUNG
   ========================================================== */
$name = trim($_POST['name'] ?? '');
$text = trim($_POST['text'] ?? '');

if ($name === '') $name = 'Anonym';

if (strlen($name) > MAX_NAME_LENGTH) {
    abortKommentar('Name zu lang.', $returnUrl);
}

if (preg_match('/(https?:\/\/|www\.|\.com|\.de|\.org)/i', $name)) {
    abortKommentar('Links im Namen nicht erlaubt.', $returnUrl);
}

if (strlen($text) < MIN_COMMENT_LENGTH) {
    abortKommentar('Kommentar zu kurz.', $returnUrl);
}

if (strlen($text) > MAX_COMMENT_LENGTH) {
    abortKommentar('Kommentar zu lang.', $returnUrl);
}

/* ==========================================================
   5. CONTENT-SPAM
   ========================================================== */
if (preg_match('/(https?:\/\/|www\.|\.com|\.de|\.org|\.net|\.ch|\.at)/i', $text)) {
    abortKommentar('Links im Kommentar nicht erlaubt.', $returnUrl);
}

foreach ($blockedTLDs as $tld) {
    if (stripos($text, $tld) !== false || stripos($name, $tld) !== false) {
        abortKommentar('Ungültiger Inhalt.', $returnUrl);
    }
}

if (BLOCK_NON_LATIN && preg_match('/[\x{0400}-\x{04FF}\x{4E00}-\x{9FFF}]/u', $text.$name)) {
    abortKommentar('Nur lateinische Zeichen erlaubt.', $returnUrl);
}

$spamKeywords = array_merge([
    'viagra','casino','lottery','buy now','click here',
    'earn money','work from home','bitcoin','crypto'
], $customSpamKeywords ?? []);

foreach ($spamKeywords as $kw) {
    if (stripos($text, $kw) !== false) {
        abortKommentar('Spam erkannt.', $returnUrl);
    }
}

if (preg_match('/(.)\1{9,}/', $text)) {
    abortKommentar('Ungültiger Text.', $returnUrl);
}

if (preg_match('/^[A-Z\s!]+$/', $text) && strlen($text) > 20) {
    abortKommentar('Bitte normale Schreibweise verwenden.', $returnUrl);
}

/* ==========================================================
   6. USER-AGENT / LAND
   ========================================================== */
$ua = $_SERVER['HTTP_USER_AGENT'] ?? '';
foreach (['bot','crawler','spider','curl','wget','python'] as $b) {
    if (stripos($ua, $b) !== false) {
        abortKommentar('Zugriff verweigert.', $returnUrl);
    }
}

if (!empty($_SERVER['HTTP_ACCEPT_LANGUAGE'])) {
    foreach ($blockedCountries as $c) {
        if (stripos($_SERVER['HTTP_ACCEPT_LANGUAGE'], $c) === 0) {
            abortKommentar('Region blockiert.', $returnUrl);
        }
    }
}

/* ==========================================================
   7. DUPLIKAT (24h)
   ========================================================== */
$stmt = $db->prepare("
  SELECT COUNT(*) FROM kommentare
  WHERE text = ? AND zeit > datetime('now','-24 hours')
");
$stmt->execute([$text]);
if ($stmt->fetchColumn() > 0) {
    abortKommentar('Kommentar bereits gesendet.', $returnUrl);
}

/* ==========================================================
   8. INSERT
   ========================================================== */
$ipHash = hash('sha256', ($_SERVER['REMOTE_ADDR'] ?? '') . SALT);
$uaHash = hash('sha256', $ua);

$cols = array_column(
    $db->query("PRAGMA table_info(kommentare)")->fetchAll(PDO::FETCH_ASSOC),
    'name'
);

$sql = "INSERT INTO kommentare (ziel,name,text,zeit";
$ph  = "?,?,?,datetime('now')";
$val = [$ziel,$name,$text];

if (in_array('ip_hash',$cols)) {
    $sql .= ",ip_hash"; $ph .= ",?"; $val[] = $ipHash;
}
if (in_array('user_agent_hash',$cols)) {
    $sql .= ",user_agent_hash"; $ph .= ",?"; $val[] = $uaHash;
}

$sql .= ") VALUES ($ph)";
$db->prepare($sql)->execute($val);

/* ==========================================================
   9. MAIL (mail())
   ========================================================== */
if (EMAIL_ENABLED && !empty(ADMIN_EMAIL)) {

    $schema = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
    $link   = $schema . '://' . $_SERVER['HTTP_HOST'] . $returnUrl;

    $betreff = "Neuer Kommentar: $ziel" . ($version ? " ($version)" : "");

    $mailText =
        "Neuer Kommentar\n\n".
        "Ziel: $ziel\n".
        ($version ? "Version: $version\n" : '').
        "Name: $name\n\n".
        "Kommentar:\n$text\n\n".
        "Link:\n$link\n";

    $headers  = "From: Kommentar-System <noreply@" . $_SERVER['HTTP_HOST'] . ">\r\n";
    $headers .= "Content-Type: text/plain; charset=UTF-8";

    @mail(ADMIN_EMAIL, $betreff, $mailText, $headers);
}

/* ==========================================================
   SMTP / PHPMailer – KOMPLETT VORBEREITET (AUS)
   ========================================================== */
/*
use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

require_once __DIR__.'/PHPMailer/src/PHPMailer.php';
require_once __DIR__.'/PHPMailer/src/SMTP.php';
require_once __DIR__.'/PHPMailer/src/Exception.php';

$mail = new PHPMailer(true);
$mail->isSMTP();
$mail->Host = SMTP_HOST;
$mail->SMTPAuth = true;
$mail->Username = SMTP_USER;
$mail->Password = SMTP_PASS;
$mail->Port = SMTP_PORT;
$mail->SMTPSecure = SMTP_PORT == 465
    ? PHPMailer::ENCRYPTION_SMTPS
    : PHPMailer::ENCRYPTION_STARTTLS;

$mail->CharSet = 'UTF-8';
$mail->setFrom('noreply@'.$_SERVER['HTTP_HOST'],'Kommentar-System');
$mail->addAddress(ADMIN_EMAIL);
$mail->Subject = $betreff;
$mail->Body = $mailText;
$mail->send();
*/

/* ==========================================================
   10. FERTIG
   ========================================================== */
$_SESSION['csrf_token'] = bin2hex(random_bytes(32));
header('Location: ' . $returnUrl);
exit;
