<?php
require_once __DIR__ . '/FileClientInterface.php';

class FtpClient implements FileClientInterface
{
    private $host;
    private $user;
    private $pass;
    private $port;
    private $conn;

    public function __construct(string $host, string $user, string $pass, int $port = 21)
    {
        $this->host = $host;
        $this->user = $user;
        $this->pass = $pass;
        $this->port = $port;
    }

    public function connect()
    {
        if ($this->conn) return;

        $this->conn = ftp_connect($this->host, $this->port, 10);
        if (!$this->conn) {
            throw new Exception("FTP-Verbindung fehlgeschlagen");
        }

        if (!ftp_login($this->conn, $this->user, $this->pass)) {
            throw new Exception("FTP-Login fehlgeschlagen");
        }

        ftp_pasv($this->conn, true);
    }

    /** LISTING MIT GRÖSSE & DATUM */
    public function listDirectory(string $path): array
    {
        $this->connect();
        if ($path === '') $path = '.';

        // Versuch 1: MLSD (modern & strukturiert)
        $raw = @ftp_mlsd($this->conn, $path);
        
        if (is_array($raw) && count($raw) > 0) {
            $result = [];
            foreach ($raw as $entry) {
                if (($entry['name'] ?? '') === '.' || ($entry['name'] ?? '') === '..') {
                    continue;
                }
                
                $result[] = [
                    'name'   => $entry['name'] ?? '',
                    'is_dir' => ($entry['type'] ?? '') === 'dir',
                    'size'   => (int)($entry['size'] ?? 0),
                    'mtime'  => isset($entry['modify']) ? strtotime($entry['modify']) : time()
                ];
            }
            return $result;
        }

        // Fallback: rawlist (klassisch)
        $raw = ftp_rawlist($this->conn, $path);
        if (!is_array($raw)) return [];

        $result = [];

        foreach ($raw as $line) {
            // Beispiel:
            // -rw-r--r-- 1 user group  512 Dec  1 12:30 file.txt
            // drwxr-xr-x 2 user group 4096 Nov 15 2024 folder
            
            $parts = preg_split('/\s+/', trim($line), 9);
            if (count($parts) < 9) continue;

            $permissions = $parts[0];
            $name        = $parts[8];
            
            // . und .. überspringen
            if ($name === '.' || $name === '..') continue;

            $isDir = ($permissions[0] ?? '') === 'd';
            $size  = (int)$parts[4];
            
            // Datum parsen
            $month       = $parts[5];
            $day         = $parts[6];
            $timeOrYear  = $parts[7];
            $mtime       = $this->parseDate($month, $day, $timeOrYear);

            $result[] = [
                'name'   => $name,
                'is_dir' => $isDir,
                'size'   => $isDir ? 0 : $size,
                'mtime'  => $mtime
            ];
        }

        return $result;
    }

    /** Datum aus FTP-Format berechnen */
    private function parseDate($month, $day, $timeOrYear): int
    {
        // Monatsnamen in Zahlen
        $months = [
            'jan' => 1, 'feb' => 2, 'mar' => 3, 'apr' => 4,
            'may' => 5, 'jun' => 6, 'jul' => 7, 'aug' => 8,
            'sep' => 9, 'oct' => 10, 'nov' => 11, 'dec' => 12
        ];
        
        $monthLower = strtolower(substr($month, 0, 3));
        $monthNum = $months[$monthLower] ?? date('n');
        
        if (str_contains($timeOrYear, ':')) {
            // Format: "12:30" -> aktuelles Jahr
            $year = date('Y');
            list($hour, $minute) = explode(':', $timeOrYear);
            return mktime((int)$hour, (int)$minute, 0, $monthNum, (int)$day, (int)$year);
        } else {
            // Format: "2024" -> Jahr gegeben
            $year = (int)$timeOrYear;
            return mktime(0, 0, 0, $monthNum, (int)$day, $year);
        }
    }

    public function isDir(string $path): bool
    {
        $this->connect();
        $list = ftp_rawlist($this->conn, $path);
        if (!$list) return false;
        return str_starts_with($list[0], 'd');
    }

    public function getFileContents(string $path)
    {
        $this->connect();

        $tmp = tmpfile();
        $meta = stream_get_meta_data($tmp);
        $local = $meta['uri'];

        if (!ftp_get($this->conn, $local, $path, FTP_BINARY)) {
            return false;
        }

        return file_get_contents($local);
    }

    public function putFileContents(string $path, string $data): bool
    {
        $this->connect();

        $tmp = tmpfile();
        $meta = stream_get_meta_data($tmp);
        $local = $meta['uri'];
        file_put_contents($local, $data);

        return ftp_put($this->conn, $path, $local, FTP_BINARY);
    }

    public function delete(string $path): bool
    {
        $this->connect();
        return ftp_delete($this->conn, $path);
    }

    public function mkdir(string $path): bool
    {
        $this->connect();
        return ftp_mkdir($this->conn, $path) !== false;
    }

    public function rename(string $old, string $new): bool
    {
        $this->connect();
        return ftp_rename($this->conn, $old, $new);
    }

    public function chmod(string $path, int $mode): bool
    {
        $this->connect();

        if (function_exists('ftp_chmod')) {
            return ftp_chmod($this->conn, $mode, $path) !== false;
        }

        return false;
    }
}