<?php
require_once __DIR__ . '/FileClientInterface.php';
require_once __DIR__ . '/../vendor/autoload.php';

use phpseclib3\Net\SFTP;

class SftpClient implements FileClientInterface
{
    private $host;
    private $user;
    private $pass;
    private $port;
    private $sftp;

    public function __construct(string $host, string $user, string $pass, int $port = 22)
    {
        $this->host = $host;
        $this->user = $user;
        $this->pass = $pass;
        $this->port = $port;
    }

    public function connect()
    {
        if ($this->sftp) return;

        $this->sftp = new SFTP($this->host, $this->port, 10);

        if (!$this->sftp->login($this->user, $this->pass)) {
            throw new Exception("SFTP-Login fehlgeschlagen");
        }
    }

    public function listDirectory(string $path): array
    {
        $this->connect();
        if ($path === '') $path = '.';

        $raw = $this->sftp->rawlist($path);

        if (!is_array($raw)) return [];

        $result = [];

        foreach ($raw as $name => $info) {
            if ($name === '.' || $name === '..') continue;

            $isDir = isset($info['type']) && $info['type'] == SFTP::FILE_TYPE_DIRECTORY;

            $size  = $info['size']  ?? 0;
            $mtime = $info['mtime'] ?? 0;

            $result[] = [
                'name'   => $name,
                'is_dir' => $isDir,
                'size'   => $isDir ? 0 : $size,
                'mtime'  => $mtime,
            ];
        }
        return $result;
    }

    public function isDir(string $path): bool
    {
        $this->connect();

        $info = $this->sftp->stat($path);
        if (!isset($info['type'])) return false;

        return $info['type'] == SFTP::FILE_TYPE_DIRECTORY;
    }

    public function getFileContents(string $path)
    {
        $this->connect();
        return $this->sftp->get($path);
    }

    public function putFileContents(string $path, string $data): bool
    {
        $this->connect();
        return $this->sftp->put($path, $data);
    }

    public function delete(string $path): bool
    {
        $this->connect();
        return $this->sftp->delete($path);
    }

    public function mkdir(string $path): bool
    {
        $this->connect();
        return $this->sftp->mkdir($path);
    }

    public function rename(string $old, string $new): bool
    {
        $this->connect();
        return $this->sftp->rename($old, $new);
    }

    public function chmod(string $path, int $mode): bool
    {
        $this->connect();
        return $this->sftp->chmod($mode, $path);
    }
}
