<?php

session_start();

// Events laden
$events_file = "events.json";
$events = json_decode(file_get_contents($events_file), true) ?? [];

// Event hinzufügen/bearbeiten
if (isset($_POST['save_event'])) {
    $new_event = [
        'id' => $_POST['event_id'] ?: uniqid(),
        'datum' => $_POST['datum'],
        'titel' => $_POST['titel'],
        'uhrzeit' => $_POST['uhrzeit'],
        'beschreibung' => $_POST['beschreibung'],
        'farbe' => $_POST['farbe'] ?: 'primary'
    ];
    
    // Prüfen ob Event existiert (bearbeiten)
    $found = false;
    foreach ($events as $key => $event) {
        if ($event['id'] === $new_event['id']) {
            $events[$key] = $new_event;
            $found = true;
            break;
        }
    }
    
    // Neues Event hinzufügen
    if (!$found) {
        $events[] = $new_event;
    }
    
    // Events nach Datum sortieren
    usort($events, function($a, $b) {
        return strtotime($a['datum']) - strtotime($b['datum']);
    });
    
    file_put_contents($events_file, json_encode($events, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    $success_message = "Event erfolgreich gespeichert!";
}

// Event löschen
if (isset($_POST['delete_event'])) {
    $delete_id = $_POST['delete_id'];
    $events = array_filter($events, function($event) use ($delete_id) {
        return $event['id'] !== $delete_id;
    });
    file_put_contents($events_file, json_encode(array_values($events), JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    $success_message = "Event erfolgreich gelöscht!";
}

// Event für Bearbeitung laden
$edit_event = null;
if (isset($_GET['edit'])) {
    $edit_id = $_GET['edit'];
    foreach ($events as $event) {
        if ($event['id'] === $edit_id) {
            $edit_event = $event;
            break;
        }
    }
}
?>
<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Panel - Vereinskalender</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        .event-row:hover {
            background-color: #f8f9fa;
        }
        
        .event-card {
            border-left: 4px solid;
            margin-bottom: 1rem;
        }
        
        .event-date {
            font-weight: bold;
            color: #0d6efd;
        }
        
        .event-time {
            font-size: 0.9rem;
            color: #6c757d;
        }
        
        .event-title {
            font-weight: bold;
            margin-bottom: 0.25rem;
        }
        
        .event-description {
            font-size: 0.9rem;
            color: #6c757d;
        }
        
        .action-buttons {
            display: flex;
            gap: 0.5rem;
            flex-wrap: wrap;
        }
        
        /* Mobile-spezifische Styles */
        @media (max-width: 768px) {
            .container {
                padding-left: 10px;
                padding-right: 10px;
            }
            
            .card {
                margin-bottom: 1rem;
            }
            
            .btn {
                font-size: 0.9rem;
            }
            
            .form-label {
                font-weight: 600;
                margin-bottom: 0.25rem;
            }
            
            .form-control, .form-select {
                font-size: 1rem;
            }
            
            /* Event-Liste als Karten auf Mobile */
            .mobile-event-list {
                display: block;
            }
            
            .desktop-table {
                display: none;
            }
            
            .event-card {
                padding: 1rem;
                background: white;
                border-radius: 0.5rem;
                box-shadow: 0 2px 4px rgba(0,0,0,0.1);
            }
            
            .header-actions {
                flex-direction: column;
                gap: 1rem;
                text-align: center;
            }
            
            .header-actions h2 {
                order: 1;
                margin-bottom: 0;
            }
            
            .header-actions .btn {
                order: 2;
                align-self: stretch;
            }
        }
        
        @media (min-width: 769px) {
            .mobile-event-list {
                display: none;
            }
            
            .desktop-table {
                display: block;
            }
        }
        
        /* Sehr kleine Bildschirme */
        @media (max-width: 480px) {
            .action-buttons {
                justify-content: center;
            }
            
            .action-buttons .btn {
                flex: 1;
                min-width: 80px;
            }
            
            .event-card {
                padding: 0.75rem;
            }
            
            h2 {
                font-size: 1.5rem;
            }
            
            .card-header h5 {
                font-size: 1.1rem;
            }
        }
    </style>
</head>
<body class="bg-light">
<div class="container mt-4">
    
    <div class="d-flex justify-content-between align-items-center mb-4 header-actions">
        <h2>Kalender Admin</h2>
        <div>
            <a href="kalender.php" class="btn btn-secondary">📅 Zum Kalender</a>
        </div>
    </div>

    <?php if (isset($success_message)): ?>
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            <?= $success_message ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    <?php endif; ?>

    <div class="row">
        <div class="col-lg-4 col-md-5">
            <div class="card sticky-top" style="top: 1rem;">
                <div class="card-header">
                    <h5 class="mb-0"><?= $edit_event ? '✏️ Event bearbeiten' : '➕ Neues Event' ?></h5>
                </div>
                <div class="card-body">
                    <form method="POST">
                        <input type="hidden" name="event_id" value="<?= $edit_event['id'] ?? '' ?>">
                        
                        <div class="mb-3">
                            <label class="form-label">📅 Datum *</label>
                            <input type="date" name="datum" class="form-control" 
                                   value="<?= $edit_event['datum'] ?? '' ?>" required>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">📝 Titel *</label>
                            <input type="text" name="titel" class="form-control" 
                                   value="<?= htmlspecialchars($edit_event['titel'] ?? '') ?>" 
                                   placeholder="Event-Titel eingeben..." required>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">⏰ Uhrzeit</label>
                            <input type="time" name="uhrzeit" class="form-control" 
                                   value="<?= $edit_event['uhrzeit'] ?? '' ?>">
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">📄 Beschreibung</label>
                            <textarea name="beschreibung" class="form-control" rows="3" 
                                      placeholder="Optionale Beschreibung..."><?= htmlspecialchars($edit_event['beschreibung'] ?? '') ?></textarea>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">🎨 Farbe</label>
                            <div class="d-flex align-items-center">
                                <select name="farbe" id="farbe" class="form-select me-2">
                                    <?php
                                    $farben = [
                                        'primary' => ['name' => 'Blau (Standard)', 'color' => '#0d6efd'],
                                        'success' => ['name' => 'Grün', 'color' => '#198754'],
                                        'danger' => ['name' => 'Rot', 'color' => '#dc3545'],
                                        'warning' => ['name' => 'Orange', 'color' => '#fd7e14'],
                                        'info' => ['name' => 'Hellblau', 'color' => '#0dcaf0'],
                                        'secondary' => ['name' => 'Grau', 'color' => '#6c757d'],
                                        'dark' => ['name' => 'Dunkel', 'color' => '#212529'],
                                        'purple' => ['name' => 'Lila', 'color' => '#6f42c1'],
                                        'pink' => ['name' => 'Rosa', 'color' => '#d63384'],
                                        'teal' => ['name' => 'Türkis', 'color' => '#20c997']
                                    ];
                                    
                                    $selected_color = $edit_event['farbe'] ?? 'primary';
                                    foreach ($farben as $key => $farbe):
                                    ?>
                                        <option value="<?= $key ?>" 
                                                data-color="<?= htmlspecialchars($farbe['color']) ?>"
                                                <?= $selected_color === $key ? 'selected' : '' ?>>
                                            <?= $farbe['name'] ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                                <span id="color-preview" style="width: 2rem; height: 2rem; border-radius: 0.25rem; border: 1px solid #ccc;"></span>
                            </div>
                        </div>
                        
                        <div class="d-grid gap-2">
                            <button type="submit" name="save_event" class="btn btn-primary">
                                <?= $edit_event ? '💾 Änderungen speichern' : '➕ Event hinzufügen' ?>
                            </button>
                            <?php if ($edit_event): ?>
                                <a href="kalender_admin.php" class="btn btn-secondary">❌ Abbrechen</a>
                            <?php endif; ?>
                        </div>
                    </form>
                </div>
            </div>
        </div>

        <div class="col-lg-8 col-md-7">
            <div class="card">
                <div class="card-header">
                    <h5 class="mb-0">📋 Alle Events (<?= count($events) ?>)</h5>
                </div>
                <div class="card-body p-0">
                    <?php if (empty($events)): ?>
                        <div class="p-4 text-center text-muted">
                            <div class="mb-2">📅</div>
                            <div>Noch keine Events vorhanden.</div>
                            <small>Fügen Sie Ihr erstes Event hinzu!</small>
                        </div>
                    <?php else: ?>
                        <div class="desktop-table">
                            <div class="table-responsive">
                                <table class="table table-hover mb-0">
                                    <thead class="table-light">
                                        <tr>
                                            <th>Datum</th>
                                            <th>Zeit</th>
                                            <th>Titel</th>
                                            <th>Farbe</th>
                                            <th>Beschreibung</th>
                                            <th width="120">Aktionen</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($events as $event): ?>
                                            <tr class="event-row">
                                                <td>
                                                    <strong><?= date("d.m.Y", strtotime($event['datum'])) ?></strong>
                                                </td>
                                                <td>
                                                    <?= $event['uhrzeit'] ? $event['uhrzeit'] . ' Uhr' : '-' ?>
                                                </td>
                                                <td>
                                                    <strong><?= htmlspecialchars($event['titel']) ?></strong>
                                                </td>
                                                <td>
                                                    <?php 
                                                    $event_farbe = $event['farbe'] ?? 'primary';
                                                    $farben = [
                                                        'primary' => '#0d6efd', 'success' => '#198754', 'danger' => '#dc3545',
                                                        'warning' => '#fd7e14', 'info' => '#0dcaf0', 'secondary' => '#6c757d',
                                                        'dark' => '#212529', 'purple' => '#6f42c1', 'pink' => '#d63384', 'teal' => '#20c997'
                                                    ];
                                                    ?>
                                                    <span class="badge" style="background-color: <?= $farben[$event_farbe] ?? $farben['primary'] ?>;">
                                                        <?= ucfirst($event_farbe) ?>
                                                    </span>
                                                </td>
                                                <td>
                                                    <?php 
                                                    $desc = htmlspecialchars($event['beschreibung']);
                                                    echo strlen($desc) > 50 ? substr($desc, 0, 50) . '...' : $desc;
                                                    ?>
                                                </td>
                                                <td>
                                                    <div class="action-buttons">
                                                        <a href="?edit=<?= $event['id'] ?>" class="btn btn-sm btn-outline-primary" title="Bearbeiten">
                                                            ✏️
                                                        </a>
                                                        <form method="POST" style="display: inline;" 
                                                              onsubmit="return confirm('Event wirklich löschen?')">
                                                            <input type="hidden" name="delete_id" value="<?= $event['id'] ?>">
                                                            <button type="submit" name="delete_event" class="btn btn-sm btn-outline-danger" title="Löschen">
                                                                🗑️
                                                            </button>
                                                        </form>
                                                    </div>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>

                        <div class="mobile-event-list p-3">
                            <?php 
                            $farben = [
                                'primary' => '#0d6efd', 'success' => '#198754', 'danger' => '#dc3545',
                                'warning' => '#fd7e14', 'info' => '#0dcaf0', 'secondary' => '#6c757d',
                                'dark' => '#212529', 'purple' => '#6f42c1', 'pink' => '#d63384', 'teal' => '#20c997'
                            ];
                            
                            foreach ($events as $event): 
                                $event_farbe = $event['farbe'] ?? 'primary';
                                $border_color = $farben[$event_farbe] ?? $farben['primary'];
                            ?>
                                <div class="event-card" style="border-left-color: <?= $border_color ?>;">
                                    <div class="d-flex justify-content-between align-items-start mb-2">
                                        <div class="event-date">
                                            📅 <?= date("d.m.Y", strtotime($event['datum'])) ?>
                                        </div>
                                        <span class="badge" style="background-color: <?= $border_color ?>;">
                                            <?= ucfirst($event_farbe) ?>
                                        </span>
                                    </div>
                                    
                                    <?php if ($event['uhrzeit']): ?>
                                        <div class="event-time mb-2">
                                            ⏰ <?= $event['uhrzeit'] ?> Uhr
                                        </div>
                                    <?php endif; ?>
                                    
                                    <div class="event-title">
                                        <?= htmlspecialchars($event['titel']) ?>
                                    </div>
                                    
                                    <?php if ($event['beschreibung']): ?>
                                        <div class="event-description mb-3">
                                            <?= nl2br(htmlspecialchars($event['beschreibung'])) ?>
                                        </div>
                                    <?php endif; ?>
                                    
                                    <div class="action-buttons">
                                        <a href="?edit=<?= $event['id'] ?>" class="btn btn-sm btn-outline-primary">
                                            ✏️ Bearbeiten
                                        </a>
                                        <form method="POST" style="display: inline;" 
                                              onsubmit="return confirm('Event wirklich löschen?')">
                                            <input type="hidden" name="delete_id" value="<?= $event['id'] ?>">
                                            <button type="submit" name="delete_event" class="btn btn-sm btn-outline-danger">
                                                🗑️ Löschen
                                            </button>
                                        </form>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>

</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
<script>
// Auto-Hide Success Messages
document.addEventListener('DOMContentLoaded', function() {
    const alerts = document.querySelectorAll('.alert-success');
    alerts.forEach(alert => {
        setTimeout(() => {
            if (alert) {
                alert.classList.add('fade');
                setTimeout(() => alert.remove(), 300);
            }
        }, 3000);
    });
});

// Form Validation Enhancement
document.querySelector('form').addEventListener('submit', function(e) {
    const titel = document.querySelector('input[name="titel"]').value.trim();
    const datum = document.querySelector('input[name="datum"]').value;
    
    if (!titel) {
        alert('Bitte geben Sie einen Event-Titel ein.');
        e.preventDefault();
        return;
    }
    
    if (!datum) {
        alert('Bitte wählen Sie ein Datum aus.');
        e.preventDefault();
        return;
    }
});

// === FARB-VORSCHAU-LOGIK ===
document.addEventListener('DOMContentLoaded', function() {
    const colorSelect = document.getElementById('farbe');
    const colorPreview = document.getElementById('color-preview');

    function updateColorPreview() {
        const selectedOption = colorSelect.options[colorSelect.selectedIndex];
        const color = selectedOption.getAttribute('data-color');
        if (color) {
            colorPreview.style.backgroundColor = color;
        }
    }

    // Initialisieren der Vorschau beim Laden der Seite
    updateColorPreview();

    // Event-Listener für Änderungen im Dropdown
    colorSelect.addEventListener('change', updateColorPreview);
});
</script>
</body>
</html>