<?php
/**
 * api.php – Zentrale Logik für die Linkliste
 * - absoluter Pfad
 * - atomisches Schreiben
 * - fehlertolerantes JSON-Laden
 * - Debug-Ausgaben möglich
 */

/* =========================
   KONFIGURATION
========================= */

define('DATA_FILE', __DIR__ . '/data.json');
define('DEBUG_FILE', __DIR__ . '/debug.txt');

$adminEmail = 'volker@niederastroth.de';
$siteName   = 'Linkliste';

$fromEmail = 'no-reply@' . ($_SERVER['HTTP_HOST'] ?? 'localhost');
$headers  = "From: $siteName <$fromEmail>\r\n";
$headers .= "Reply-To: $fromEmail\r\n";
$headers .= "Content-Type: text/plain; charset=UTF-8\r\n";

/* =========================
   HILFSFUNKTIONEN
========================= */

function debug_log($text) {
    file_put_contents(DEBUG_FILE, date('H:i:s') . ' - ' . $text . "\n", FILE_APPEND);
}

function lade_daten() {
    if (!file_exists(DATA_FILE)) {
        return [
            'links' => [],
            'categories' => [],
            'headerColor' => '#050000'
        ];
    }

    $json = file_get_contents(DATA_FILE);

    // BOM entfernen
    $json = preg_replace('/^\xEF\xBB\xBF/', '', $json);

    $data = json_decode($json, true);

    if (!is_array($data)) {
        debug_log('JSON ungültig – Fallback benutzt');
        return [
            'links' => [],
            'categories' => [],
            'headerColor' => '#050000'
        ];
    }

    return $data;
}

function speichere_daten(array $data) {
    $tmp = DATA_FILE . '.tmp';

    $json = json_encode(
        $data,
        JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE
    );

    if ($json === false) {
        debug_log('json_encode fehlgeschlagen');
        return false;
    }

    if (file_put_contents($tmp, $json) === false) {
        debug_log('Konnte TMP-Datei nicht schreiben');
        return false;
    }

    if (!rename($tmp, DATA_FILE)) {
        debug_log('rename() fehlgeschlagen');
        return false;
    }

    return true;
}

/* =========================
   INITIALISIERUNG
========================= */

if (!file_exists(DATA_FILE)) {
    speichere_daten([
        'links' => [],
        'categories' => [
            'Allgemein',
            'Technologie',
            'Design',
            'Business',
            'Bildung',
            'Unterhaltung',
            'Private Webseite',
            'PHP',
            'Verein',
            'Tutorials',
            'Mobirise'
        ],
        'headerColor' => '#050000'
    ]);
}

/* =========================
   ZENTRALE AKTION
========================= */

function link_hinzufuegen(array $daten) {
    global $adminEmail, $siteName, $headers;

    debug_log('START link_hinzufuegen');

    if (!is_writable(DATA_FILE)) {
        debug_log('data.json NICHT beschreibbar');
        return false;
    }

    $data = lade_daten();
    debug_log('JSON geladen, Einträge: ' . count($data['links']));

    if (!in_array($daten['category'], $data['categories'])) {
        $data['categories'][] = $daten['category'];
        debug_log('Kategorie ergänzt: ' . $daten['category']);
    }

    $data['links'][] = [
        'id' => time(),
        'title' => $daten['title'],
        'url' => $daten['url'],
        'description' => $daten['description'],
        'category' => $daten['category'],
        'submitter' => $daten['submitter'],
        'status' => 'pending',
        'dateSubmitted' => date(DATE_ATOM)
    ];

    debug_log('Link angehängt');

    if (!speichere_daten($data)) {
        debug_log('Speichern fehlgeschlagen');
        return false;
    }

    debug_log('JSON erfolgreich gespeichert');

    // Mail
    @mail(
        $adminEmail,
        "Neuer Link – $siteName",
        "Titel: {$daten['title']}\nURL: {$daten['url']}",
        $headers
    );

    debug_log('Mail gesendet');
    return true;
}

/* =========================
   API-DIREKTAUFRUF?
========================= */

if (basename($_SERVER['SCRIPT_FILENAME']) !== basename(__FILE__)) {
    return; // nur Include
}

/* =========================
   API-OUTPUT
========================= */

header('Content-Type: application/json; charset=utf-8');

$action = $_GET['action'] ?? '';
$input  = json_decode(file_get_contents('php://input'), true);

switch ($action) {

    case 'get':
        echo json_encode(lade_daten());
        break;

    case 'addLink':
        link_hinzufuegen([
            'title'       => trim($input['title'] ?? ''),
            'url'         => trim($input['url'] ?? ''),
            'description' => trim($input['description'] ?? ''),
            'category'    => trim($input['category'] ?? 'Allgemein'),
            'submitter'   => trim($input['submitter'] ?? 'Anonym')
        ]);
        echo json_encode(['success' => true]);
        break;

    default:
        http_response_code(400);
        echo json_encode(['error' => 'Invalid action']);
}
