<?php
declare(strict_types=1);

function base_url(string $path = ''): string {
  $https = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off');
  $proto = $https ? 'https://' : 'http://';
  $host  = $_SERVER['HTTP_HOST'] ?? 'localhost';
  $p = ltrim($path, '/');
  return $proto . $host . '/' . $p;
}

function ensure_upload_dir(): void {
  if (!is_dir(UPLOAD_DIR)) {
    mkdir(UPLOAD_DIR, 0755, true);
  }
}

function e(string $s): string {
  return htmlspecialchars($s, ENT_QUOTES, 'UTF-8');
}

function slugify(string $s): string {
  $s = trim(mb_strtolower($s, 'UTF-8'));
  $s = preg_replace('~[^a-z0-9äöüß]+~u', '-', $s);
  $s = trim($s, '-');
  return $s ?: 'post';
}

/* =====================================================
   🔥 DEUTSCHE DATUMS-HILFSFUNKTIONEN
   (ohne setlocale / Intl – 100% stabil)
===================================================== */

/**
 * Gibt deutschen Monatsnamen zurück
 * 1 → Januar, 12 → Dezember
 */
function monat_de(int $monat): string {
  static $monate = [
    1  => 'Januar',
    2  => 'Februar',
    3  => 'März',
    4  => 'April',
    5  => 'Mai',
    6  => 'Juni',
    7  => 'Juli',
    8  => 'August',
    9  => 'September',
    10 => 'Oktober',
    11 => 'November',
    12 => 'Dezember',
  ];
  return $monate[$monat] ?? '';
}

/**
 * Format: 24. Dezember 2025
 */
function datum_de(int $timestamp): string {
  return date('d.', $timestamp) . ' ' .
         monat_de((int)date('n', $timestamp)) . ' ' .
         date('Y', $timestamp);
}

/**
 * Format: Dezember 2025 (für Archiv)
 */
function monat_jahr_de(int $timestamp): string {
  return monat_de((int)date('n', $timestamp)) . ' ' . date('Y', $timestamp);
}
/* =====================================================
   🔎 BLOG-SPOILER / EXCERPT
===================================================== */

/**
 * Erstellt einen kurzen Vorschautext aus HTML-Inhalt
 *
 * @param string $html
 * @param int    $words
 * @return string
 */
function blog_excerpt(string $html, int $words = 30): string {
    $text = trim(strip_tags($html));
    if ($text === '') {
        return '';
    }

    $parts = preg_split('/\s+/u', $text);
    if (!$parts || count($parts) <= $words) {
        return $text;
    }

    return implode(' ', array_slice($parts, 0, $words)) . ' …';
}
/**
 * Zählt Aufrufe eines Artikels über JSON-Datei
 */
function count_post_view(string $slug): int {

    $file = __DIR__ . '/../data/views.json';

    // Datei laden oder initialisieren
    if (file_exists($file)) {
        $data = json_decode(file_get_contents($file), true);
        if (!is_array($data)) {
            $data = [];
        }
    } else {
        $data = [];
    }

    // Zähler erhöhen
    if (!isset($data[$slug])) {
        $data[$slug] = 0;
    }
    $data[$slug]++;

    // Sicher speichern
    file_put_contents(
        $file,
        json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE),
        LOCK_EX
    );

    return $data[$slug];
}
