<?php
declare(strict_types=1);

require_once __DIR__ . '/../inc/config.php';
require_once __DIR__ . '/../inc/db.php';
require_once __DIR__ . '/../inc/helpers.php';

$pdo = db();

/* =========================================================
   AUFRUFZÄHLER (ohne DB) – speichert in /mobiblog/data/views.json
========================================================= */

function mobiblog_views_file(): string {
    $dir = __DIR__ . '/../data';
    if (!is_dir($dir)) {
        @mkdir($dir, 0755, true);
    }
    return $dir . '/views.json';
}

function mobiblog_views_read_all(): array {
    $file = mobiblog_views_file();
    if (!is_file($file)) return [];
    $json = @file_get_contents($file);
    if (!$json) return [];
    $data = json_decode($json, true);
    return is_array($data) ? $data : [];
}

function mobiblog_views_get(string $slug): int {
    $all = mobiblog_views_read_all();
    return (int)($all[$slug] ?? 0);
}

/**
 * Erhöht den Counter für $slug um 1 und gibt den neuen Wert zurück.
 * Mit Locking, damit es auch bei mehreren Zugriffen stabil bleibt.
 */
function mobiblog_views_inc(string $slug): int {
    $file = mobiblog_views_file();

    $fp = @fopen($file, 'c+'); // create if not exists
    if (!$fp) {
        // Fallback: nicht crashen
        return mobiblog_views_get($slug) + 1;
    }

    // exklusives Lock
    flock($fp, LOCK_EX);

    // lesen
    $size = filesize($file);
    $json = $size > 0 ? fread($fp, $size) : '';
    $data = $json ? json_decode($json, true) : [];
    if (!is_array($data)) $data = [];

    // erhöhen
    $data[$slug] = (int)($data[$slug] ?? 0) + 1;
    $newVal = (int)$data[$slug];

    // zurückschreiben
    rewind($fp);
    ftruncate($fp, 0);
    fwrite($fp, json_encode($data, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT));
    fflush($fp);

    flock($fp, LOCK_UN);
    fclose($fp);

    return $newVal;
}


/* --------------------
   Parameter
-------------------- */
$slug   = trim($_GET['slug']   ?? '');
$search = trim($_GET['search'] ?? '');
$year   = (int)($_GET['year']  ?? 0);
$month  = (int)($_GET['month'] ?? 0);
$page   = max(1, (int)($_GET['page'] ?? 1));
$view   = trim($_GET['view']   ?? 'cards'); // 'cards' oder 'list'

$perPage = 6;
$offset  = ($page - 1) * $perPage;

// Konfiguration für Vorschau
$excerptWordCount = 30;

/**
 * Erstellt einen Textauszug aus HTML-Content
 */
function createExcerpt(string $html, int $wordCount = 30): string {
    $text = html_entity_decode($html, ENT_QUOTES | ENT_HTML5, 'UTF-8');
    $text = strip_tags($text);
    $text = preg_replace('/\s+/u', ' ', $text);
    $text = trim($text);

    if ($text === '') {
        return '';
    }

    $words = preg_split('/\s+/u', $text);

    if (!$words || count($words) <= $wordCount) {
        return $text;
    }

    return implode(' ', array_slice($words, 0, $wordCount)) . ' …';
}

/**
 * Berechnet die geschätzte Lesezeit
 */
function estimateReadingTime(string $html): int {
    $text = strip_tags($html);
    $wordCount = str_word_count($text);
    return max(1, (int)ceil($wordCount / 200)); // 200 Wörter pro Minute
}

/**
 * Baut URL mit allen aktiven Parametern
 */
function buildUrl(array $params = []): string {
    global $search, $year, $month, $view;
    
    $defaults = [];
    if ($search) $defaults['search'] = $search;
    if ($year) $defaults['year'] = $year;
    if ($month) $defaults['month'] = $month;
    if ($view !== 'cards') $defaults['view'] = $view;
    
    $merged = array_merge($defaults, $params);
    
    return '?' . http_build_query($merged);
}

/* =========================================================
   EINZELARTIKEL
========================================================= */
if ($slug !== '') {

    $st = $pdo->prepare("
        SELECT title, slug, content, published_at
        FROM posts
        WHERE slug = ? AND status = 'published'
        LIMIT 1
    ");
    $st->execute([$slug]);
    $post = $st->fetch();

    if (!$post) {
        echo '<div class="alert alert-danger">';
        echo '<h4><i class="fa fa-exclamation-triangle"></i> Artikel nicht gefunden</h4>';
        echo '<p>Der gesuchte Artikel existiert nicht oder wurde entfernt.</p>';
        echo '<a href="?" class="btn btn-primary">Zurück zur Übersicht</a>';
        echo '</div>';
        return;
    }
// Zähler	
$viewKey = 'mobiblog_viewed_' . $slug;

if (empty($_SESSION[$viewKey])) {
    $_SESSION[$viewKey] = true;
    $views = mobiblog_views_inc($slug);
} else {
    $views = mobiblog_views_get($slug);
}



    $readTime = estimateReadingTime($post['content']);

echo '<article class="mobiblog-post">';

echo '<div class="post-header mb-4 pb-4" style="border-bottom: 2px solid #eee;">';

echo '<h1 class="display-4 mb-3">' . e($post['title']) . '</h1>';

echo '<div class="post-meta text-muted">';

echo ' 📅 ' . date('d.m.Y', (int)$post['published_at']);

echo ' &nbsp;•&nbsp; ';

echo ' • ⏱ ' . $readTime . ' Min.';

echo ' &nbsp;•&nbsp; ';

/* 👁️ VIEWS */
echo ' • 👁 ' . $views . ' Aufrufe';

echo '</div>'; // post-meta
echo '</div>'; // post-header

echo '<div class="mobiblog-content">' . $post['content'] . '</div>';

echo '<div class="post-footer mt-5 pt-4" style="border-top: 1px solid #eee;">';
echo '<a href="?" class="btn btn-outline-primary"><i class="fa fa-arrow-left"></i> Zurück zur Übersicht</a>';
echo '</div>';

echo '</article>';


    return;
}

/* =========================================================
   LISTE + SUCHE + ARCHIV
========================================================= */

$sql = "
  SELECT title, slug, content, published_at
  FROM posts
  WHERE status = 'published'
";
$params = [];

/* 🔍 Suche */
if ($search !== '') {
    $sql .= " AND (title LIKE ? OR content LIKE ?)";
    $params[] = '%' . $search . '%';
    $params[] = '%' . $search . '%';
}

/* 📅 Archiv */
if ($year > 0) {
    $sql .= " AND strftime('%Y', published_at, 'unixepoch', 'localtime') = ?";

    $params[] = (string)$year;
}

if ($month > 0) {
    $sql .= " AND strftime('%m', published_at, 'unixepoch', 'localtime') = ?";

    $params[] = str_pad((string)$month, 2, '0', STR_PAD_LEFT);
}

/* Gesamtanzahl für Pagination */
$countSql = "SELECT COUNT(*) FROM posts WHERE status = 'published'";
$countParams = [];

if ($search !== '') {
    $countSql .= " AND (title LIKE ? OR content LIKE ?)";
    $countParams[] = '%' . $search . '%';
    $countParams[] = '%' . $search . '%';
}

if ($year > 0) {
    $countSql .= " AND strftime('%Y', published_at, 'unixepoch') = ?";
    $countParams[] = (string)$year;
}

if ($month > 0) {
    $countSql .= " AND strftime('%m', published_at, 'unixepoch') = ?";
    $countParams[] = str_pad((string)$month, 2, '0', STR_PAD_LEFT);
}

$stCount = $pdo->prepare($countSql);
$stCount->execute($countParams);
$totalPosts = (int)$stCount->fetchColumn();
$totalPages = (int)ceil($totalPosts / $perPage);

/* Posts abrufen */
$sql .= " ORDER BY published_at DESC LIMIT ? OFFSET ?";
$params[] = $perPage;
$params[] = $offset;

$st = $pdo->prepare($sql);
$st->execute($params);
$posts = $st->fetchAll();

/* --------------------
   Suchformular
-------------------- */
?>
<div class="mobiblog-search-section mb-5">
    <form method="get" action="">
        <div class="input-group input-group-lg">
            <span class="input-group-addon">
                <i class="fa fa-search"></i>
            </span>
            <input type="text"
                   name="search"
                   class="form-control"
                   placeholder="Blog durchsuchen…"
                   value="<?= e($search) ?>"
                   style="border-radius: 0;">
            <?php if ($view !== 'cards'): ?>
            <input type="hidden" name="view" value="<?= e($view) ?>">
            <?php endif; ?>
            <span class="input-group-btn">
                <button class="btn btn-primary" type="submit" style="border-radius: 0;">
                    <i class="fa fa-search"></i> Suchen
                </button>
            </span>
        </div>
        <?php if ($search): ?>
        <div class="mt-3">
            <small class="text-muted">
                Suche nach: <strong>"<?= e($search) ?>"</strong> 
                <a href="?" class="ml-2 text-danger">
                    <i class="fa fa-times"></i> Zurücksetzen
                </a>
            </small>
        </div>
        <?php endif; ?>
    </form>
</div>



<?php
/* --------------------
   Ergebnisliste mit Cards
-------------------- */

if (!$posts) {
    echo '<div class="alert alert-info text-center py-5">';
    echo '<div class="mb-3"><i class="fa fa-search fa-4x text-muted"></i></div>';
    echo '<h3>Keine Beiträge gefunden</h3>';
    if ($search) {
        echo '<p class="lead">Ihre Suche nach "<strong>' . e($search) . '</strong>" ergab keine Treffer.</p>';
        echo '<a href="?" class="btn btn-primary btn-lg mt-3">Alle Beiträge anzeigen</a>';
    } else {
        echo '<p class="lead">Es sind noch keine Beiträge vorhanden.</p>';
    }
    echo '</div>';
    return;
}

// Header mit Ergebnis-Info und View-Umschalter
echo '<div class="mb-4 d-flex justify-content-between align-items-center flex-wrap">';

// Ergebnis-Info
if ($search || $year || $month) {
    echo '<div class="mb-2">';
    echo '<p class="text-muted mb-0"><i class="fa fa-info-circle"></i> ';
    echo '<strong>' . $totalPosts . '</strong> Beitrag' . ($totalPosts != 1 ? 'e' : '') . ' gefunden';
    echo '</p>';
    echo '</div>';
} else {
    echo '<div class="mb-2"></div>'; // Leer für Ausrichtung
}

// View-Umschalter
echo '<div class="btn-group mb-2" role="group" aria-label="Ansicht">';
echo '<a href="' . buildUrl(['view' => 'cards', 'page' => $page]) . '" ';
echo 'class="btn btn-sm ' . ($view === 'cards' ? 'btn-primary' : 'btn-default') . '" ';
echo 'title="Card-Ansicht">';
echo '<i class="fa fa-th"></i> Cards';
echo '</a>';
echo '<a href="' . buildUrl(['view' => 'list', 'page' => $page]) . '" ';
echo 'class="btn btn-sm ' . ($view === 'list' ? 'btn-primary' : 'btn-default') . '" ';
echo 'title="Listen-Ansicht">';
echo '<i class="fa fa-list"></i> Liste';
echo '</a>';
echo '</div>';

echo '</div>'; // Ende Header

/* =========================================================
   CARD-ANSICHT
========================================================= */
if ($view === 'cards') {
    echo '<div class="row">';

    foreach ($posts as $p) {
        $excerpt = createExcerpt($p['content'], $excerptWordCount);
        $readTime = estimateReadingTime($p['content']);
        
        echo '<div class="col-md-6 col-lg-4 mb-4">';
        echo '<div class="card h-100 shadow-sm mobiblog-card">';
        echo '<div class="card-body d-flex flex-column">';
        
        // Titel
        echo '<h4 class="card-title mb-3">';
        echo '<a href="?slug=' . e($p['slug']) . '" class="text-dark" style="text-decoration: none;">';
        echo e($p['title']);
        echo '</a>';
        echo '</h4>';
        $views = mobiblog_views_get($p['slug']);
        // Meta-Informationen
        echo '<p class="card-text text-muted mb-3" style="font-size: 0.9em;">';
        echo ' 📅 ' . date('d.m.Y', (int)$p['published_at']);
        echo ' &nbsp;•&nbsp; ';
        echo ' • ⏱ ' . $readTime . ' Min.';
		echo ' &nbsp;•&nbsp; ';
        echo ' 👁 ' . $views;
        echo '</p>';
        
        // Excerpt
        echo '<p class="card-text mobiblog-excerpt flex-grow-1">' . e($excerpt) . '</p>';
        
        // Button
        echo '<div class="mt-3">';
        echo '<a href="?slug=' . e($p['slug']) . '" class="btn btn-primary btn-block">';
        echo 'Weiterlesen <i class="fa fa-arrow-right"></i>';
        echo '</a>';
        echo '</div>';
        
        echo '</div>'; // card-body
        echo '</div>'; // card
        echo '</div>'; // col
    }

    echo '</div>'; // row
}

/* =========================================================
   LISTEN-ANSICHT
========================================================= */
else {
    echo '<div class="mobiblog-list-view">';
    
    foreach ($posts as $p) {
        $excerpt = createExcerpt($p['content'], $excerptWordCount);
        $readTime = estimateReadingTime($p['content']);
        
        echo '<div class="mobiblog-list-item mb-4 pb-4">';
        echo '<div class="row">';
        
        // Linke Spalte: Content
        echo '<div class="col-md-9">';
        echo '<h3 class="mb-2">';
        echo '<a href="?slug=' . e($p['slug']) . '" class="text-dark" style="text-decoration: none;">';
        echo e($p['title']);
        echo '</a>';
        echo '</h3>';
        $views = mobiblog_views_get($p['slug']);
        // Meta
        echo '<p class="text-muted mb-3" style="font-size: 0.9em;">';
         echo ' 📅 ' . date('d.m.Y', (int)$p['published_at']);
        echo ' &nbsp;•&nbsp; ';
         echo ' • ⏱ ' . $readTime . ' Min.';
		 echo ' &nbsp;•&nbsp; ';
         echo ' 👁 ' . $views;
        echo '</p>';
        
        // Excerpt
        echo '<p class="mobiblog-excerpt">' . e($excerpt) . '</p>';
        echo '</div>'; // col-md-9
        
        // Rechte Spalte: Button
        echo '<div class="col-md-3 d-flex align-items-center">';
        echo '<a href="?slug=' . e($p['slug']) . '" class="btn btn-primary btn-block">';
        echo 'Weiterlesen <i class="fa fa-arrow-right"></i>';
        echo '</a>';
        echo '</div>'; // col-md-3
        
        echo '</div>'; // row
        echo '</div>'; // mobiblog-list-item
    }
    
    echo '</div>'; // mobiblog-list-view
}

/* --------------------
   Pagination
-------------------- */
if ($totalPages > 1) {
    echo '<nav aria-label="Blog Pagination" class="mt-5">';
    echo '<ul class="pagination pagination-lg justify-content-center">';
    
    // Vorherige Seite
    if ($page > 1) {
        echo '<li class="page-item">';
        echo '<a class="page-link" href="' . buildUrl(['page' => $page - 1]) . '" aria-label="Vorherige">';
        echo '<span aria-hidden="true"><i class="fa fa-chevron-left"></i> Zurück</span>';
        echo '</a>';
        echo '</li>';
    } else {
        echo '<li class="page-item disabled">';
        echo '<span class="page-link"><i class="fa fa-chevron-left"></i> Zurück</span>';
        echo '</li>';
    }
    
    // Intelligente Seitenzahlen
    $range = 2;
    $showEllipsisBefore = false;
    $showEllipsisAfter = false;
    
    for ($i = 1; $i <= $totalPages; $i++) {
        if ($i == 1 || $i == $totalPages || ($i >= $page - $range && $i <= $page + $range)) {
            if ($i == $page) {
                echo '<li class="page-item active">';
                echo '<span class="page-link">' . $i . '</span>';
                echo '</li>';
            } else {
                echo '<li class="page-item">';
                echo '<a class="page-link" href="' . buildUrl(['page' => $i]) . '">' . $i . '</a>';
                echo '</li>';
            }
        } elseif ($i == $page - $range - 1 && !$showEllipsisBefore) {
            echo '<li class="page-item disabled"><span class="page-link">…</span></li>';
            $showEllipsisBefore = true;
        } elseif ($i == $page + $range + 1 && !$showEllipsisAfter) {
            echo '<li class="page-item disabled"><span class="page-link">…</span></li>';
            $showEllipsisAfter = true;
        }
    }
    
    // Nächste Seite
    if ($page < $totalPages) {
        echo '<li class="page-item">';
        echo '<a class="page-link" href="' . buildUrl(['page' => $page + 1]) . '" aria-label="Nächste">';
        echo '<span aria-hidden="true">Weiter <i class="fa fa-chevron-right"></i></span>';
        echo '</a>';
        echo '</li>';
    } else {
        echo '<li class="page-item disabled">';
        echo '<span class="page-link">Weiter <i class="fa fa-chevron-right"></i></span>';
        echo '</li>';
    }
    
    echo '</ul>';
    
    // Seiten-Info
    echo '<p class="text-center text-muted mt-3">';
    echo 'Seite <strong>' . $page . '</strong> von <strong>' . $totalPages . '</strong>';
    echo ' &nbsp;•&nbsp; ';
    echo '<strong>' . $totalPosts . '</strong> Beitrag' . ($totalPosts != 1 ? 'e' : '') . ' insgesamt';
    echo '</p>';
    
    echo '</nav>';
}
?>

<style>
/* Excerpt Styling */
.mobiblog-excerpt {
  font-family: Arial, Helvetica, sans-serif;
  font-size: 0.95em;
  line-height: 1.5;
  opacity: 0.85;
}

/* Card Hover-Effekt */
.mobiblog-card {
    border-radius: 8px;
    transition: transform 0.2s, box-shadow 0.2s;
}

.mobiblog-card:hover {
    transform: translateY(-5px);
    box-shadow: 0 10px 20px rgba(0,0,0,0.15) !important;
}

.card-title a:hover {
    color: #007bff !important;
}

/* Listen-Ansicht */
.mobiblog-list-item {
    border-bottom: 1px solid #e0e0e0;
    transition: background-color 0.2s;
    padding: 20px;
    border-radius: 8px;
}

.mobiblog-list-item:hover {
    background-color: #f8f9fa;
}

.mobiblog-list-item h3 a:hover {
    color: #007bff !important;
}

.mobiblog-list-item:last-child {
    border-bottom: none;
}

/* View Toggle Buttons */
.btn-group .btn {
    border-radius: 4px !important;
}

/* Pagination Styling */
.pagination .page-item.active .page-link {
    background-color: #007bff;
    border-color: #007bff;
}

.pagination .page-link {
    color: #007bff;
    border-radius: 4px;
    margin: 0 3px;
}

.pagination .page-link:hover {
    color: #0056b3;
}

/* Post Content Styling */
.mobiblog-content {
    line-height: 1.8;
    font-size: 1.1em;
}

.mobiblog-content img {
    max-width: 100%;
    height: auto;
    border-radius: 8px;
    margin: 20px 0;
}

/* Search Input Styling */
.mobiblog-search-section .input-group-addon {
    background-color: #f8f9fa;
    border: 1px solid #ced4da;
    border-right: none;
    padding: 0.5rem 1rem;
}

.mobiblog-search-section .form-control:focus {
    box-shadow: 0 0 0 0.2rem rgba(0,123,255,.25);
}

/* Responsive */
@media (max-width: 768px) {
    .display-4 {
        font-size: 2rem;
    }
    
    .pagination {
        font-size: 0.9rem;
    }
    
    .card-title {
        font-size: 1.1rem;
    }
    
    .mobiblog-list-item .col-md-3 {
        margin-top: 15px;
    }
    
    .d-flex.justify-content-between {
        flex-direction: column;
    }
    
    .btn-group {
        width: 100%;
    }
    
    .btn-group .btn {
        width: 50%;
    }
}
/* Suchfeld kompakter */
.mobiblog-search-section .input-group-lg .form-control {
    font-size: 0.95rem;
    padding: 0.45rem 0.75rem;
}

.mobiblog-search-section .input-group-lg .btn {
    font-size: 0.9rem;
    padding: 0.45rem 0.75rem;
}

.mobiblog-search-section .input-group-addon {
    padding: 0.45rem 0.75rem;
    font-size: 0.9rem;
}
/* Überschriften an Excerpt-Schrift anpassen */
.mobiblog-post h1,
.mobiblog-list-item h3,
.mobiblog-card h4 {
    font-family: Arial, Helvetica, sans-serif;
    font-weight: 600;
}
/* SUCHFELD – HARTER OVERRIDE */
.mobiblog-search-section .input-group.input-group-lg .form-control {
    height: 38px !important;
    font-size: 14px !important;
    padding: 6px 10px !important;
}

.mobiblog-search-section .input-group.input-group-lg .btn {
    height: 38px !important;
    font-size: 14px !important;
    padding: 6px 12px !important;
}

.mobiblog-search-section .input-group-addon {
    height: 38px !important;
    font-size: 14px !important;
    padding: 6px 10px !important;
    display: flex;
    align-items: center;
}
/* SUCHFELD – BREITE BEGRENZEN */
.mobiblog-search-section .input-group {
    max-width: 420px;   /* ← Wunschwert, z.B. 380–480 */
}
.mobiblog-search-section .form-control {
    border-left: 1px solid #ced4da;
}
/* GRAUEN KASTEN LINKS IM SUCHFELD ENTFERNEN */
.mobiblog-search-section .input-group-addon {
    display: none;
}
/* Suchbutton vertikal mittig neben dem Suchfeld */
.mobiblog-search-section .input-group {
    display: flex;
    align-items: center;
}

.mobiblog-search-section .input-group-btn {
    display: flex;
    align-items: center;
}
/* Einheitliche Schrift für Meta-Texte (Datum, Lesezeit, Views) */
.post-meta,
.card-text.text-muted,
.mobiblog-list-view .text-muted {
    font-family: Arial, Helvetica, sans-serif;
}

/* Buttons "Cards / Liste" */
.btn-group .btn {
    font-family: Arial, Helvetica, sans-serif;
}

/* Überschriften in Cards & Liste angleichen */
.card-title,
.mobiblog-list-item h3,
.mobiblog-post h1 {
    font-family: Arial, Helvetica, sans-serif;
}
/* Nur Mobiblog-Buttons */
.mobiblog-search-section .btn,
.mobiblog-card .btn,
.mobiblog-list-item .btn,
.post-footer .btn {
    font-family: Arial, Helvetica, sans-serif;
}
</style>