<?php
require_once __DIR__ . '/functions.php';
ini_set('memory_limit', '512M'); // Sicherheit für große Bilder

$cfg = read_config();
$file = basename($_GET['file'] ?? '');
$showOriginal = isset($_GET['original']);

$srcDir = normalize_path($cfg['source_dir'] ?? 'bilder');
$srcPath = $srcDir . '/' . $file;

if (!is_file($srcPath)) {
    http_response_code(404);
    exit('Bild nicht gefunden');
}

[$srcW, $srcH, $srcType] = getimagesize($srcPath);
switch ($srcType) {
    case IMAGETYPE_PNG:  $src = imagecreatefrompng($srcPath); break;
    case IMAGETYPE_JPEG: $src = imagecreatefromjpeg($srcPath); break;
    case IMAGETYPE_WEBP: $src = imagecreatefromwebp($srcPath); break;
    case IMAGETYPE_GIF:  $src = imagecreatefromgif($srcPath); break;
    default: http_response_code(415); exit('Typ nicht unterstützt.');
}

imagealphablending($src, true);
imagesavealpha($src, true);

if ($showOriginal) {
    header('Content-Type: image/jpeg');
    imagejpeg($src, null, 90);
    imagedestroy($src);
    exit;
}

// ---------------- Bild-Wasserzeichen ----------------
$type    = $cfg['watermark_type'] ?? 'text';
$opacity = (int)($cfg['opacity'] ?? 60);
$margin  = (int)($cfg['margin'] ?? 20);
$pos     = $cfg['position'] ?? 'bottom-right';

if ($type === 'image') {
    $wmRel  = $cfg['watermark_image'] ?? '';
    $wmPath = __DIR__ . '/' . ltrim($wmRel, '/');

    if (is_file($wmPath)) {
        // ✅ Cache optimiertes Wasserzeichen
        $cacheDir = __DIR__ . '/watermarks/cache';
        if (!is_dir($cacheDir)) @mkdir($cacheDir, 0777, true);
        $cacheFile = $cacheDir . '/' . md5($wmPath) . '.png';

        if (!file_exists($cacheFile)) {
            [$wW, $wH, $wT] = getimagesize($wmPath);
            $maxCache = 1200;
            $ratio = min($maxCache / $wW, $maxCache / $wH, 1);
            $newW = (int)($wW * $ratio);
            $newH = (int)($wH * $ratio);

            switch ($wT) {
                case IMAGETYPE_PNG:  $wmSrc = imagecreatefrompng($wmPath); break;
                case IMAGETYPE_JPEG: $wmSrc = imagecreatefromjpeg($wmPath); break;
                case IMAGETYPE_WEBP: $wmSrc = imagecreatefromwebp($wmPath); break;
                case IMAGETYPE_GIF:  $wmSrc = imagecreatefromgif($wmPath); break;
                default: $wmSrc = null;
            }

            if ($wmSrc) {
                $res = imagecreatetruecolor($newW, $newH);
                imagealphablending($res, false);
                imagesavealpha($res, true);
                imagecopyresampled($res, $wmSrc, 0, 0, 0, 0, $newW, $newH, $wW, $wH);
                imagepng($res, $cacheFile);
                imagedestroy($wmSrc);
                imagedestroy($res);
            }
        }

        // Cached Version verwenden
        if (file_exists($cacheFile)) $wmPath = $cacheFile;

        [$wW, $wH, $wT] = getimagesize($wmPath);
        $wm = imagecreatefrompng($wmPath);
        imagealphablending($wm, true);
        imagesavealpha($wm, true);

        $scale = max(1, min(200, (int)($cfg['image_scale'] ?? 40)));
        $tW = (int)($srcW * ($scale / 100));
        $tH = (int)($wH * ($tW / $wW));

        $res = imagecreatetruecolor($tW, $tH);
        imagealphablending($res, false);
        imagesavealpha($res, true);
        imagecopyresampled($res, $wm, 0, 0, 0, 0, $tW, $tH, $wW, $wH);
        imagedestroy($wm);

        switch ($pos) {
            case 'top-left':    $x = $margin; $y = $margin; break;
            case 'top-right':   $x = $srcW - $tW - $margin; $y = $margin; break;
            case 'bottom-left': $x = $margin; $y = $srcH - $tH - $margin; break;
            case 'center':      $x = ($srcW - $tW) / 2; $y = ($srcH - $tH) / 2; break;
            default:            $x = $srcW - $tW - $margin; $y = $srcH - $tH - $margin;
        }

        imagecopymerge($src, $res, (int)$x, (int)$y, 0, 0, $tW, $tH, $opacity);
        imagedestroy($res);
    }
}
// ---------------- Text-Wasserzeichen ----------------
else {
    $text = $cfg['watermark_text'] ?? '© Volker';
    $font = __DIR__ . '/' . ltrim($cfg['font_file'] ?? 'assets/DejaVuSans.ttf', '/');
    $fsize = (int)($cfg['font_size'] ?? 20);

    $colorHex = $cfg['color'] ?? '#FFFFFF';
    [$r, $g, $b] = sscanf($colorHex, "#%02x%02x%02x");
    $alpha = (127 - (127 * $opacity / 100));
    $color = imagecolorallocatealpha($src, $r, $g, $b, $alpha);

    $box = imagettfbbox($fsize, 0, $font, $text);
    $tw = abs($box[2] - $box[0]);
    $th = abs($box[5] - $box[3]);

    switch ($pos) {
        case 'top-left':    $x = $margin; $y = $margin + $th; break;
        case 'top-right':   $x = $srcW - $tw - $margin; $y = $margin + $th; break;
        case 'bottom-left': $x = $margin; $y = $srcH - $margin; break;
        case 'center':      $x = ($srcW - $tw) / 2; $y = ($srcH + $th) / 2; break;
        default:            $x = $srcW - $tw - $margin; $y = $srcH - $margin;
    }

    imagettftext($src, $fsize, 0, (int)$x, (int)$y, $color, $font, $text);
}

// Ausgabe
header('Content-Type: image/jpeg');
imagejpeg($src, null, 90);
imagedestroy($src);
?>
