<?php
ini_set('memory_limit', '1024M'); // bis zu 1 GB RAM erlauben
set_time_limit(0);                // kein Zeitlimit für große Bildmengen
error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once __DIR__ . '/functions.php';

$cfg = read_config();

// --- Pfade normalisieren ---
$source = normalize_path($cfg['source_dir'] ?? 'bilder');
$output = normalize_path($cfg['output_dir'] ?? 'bilder/watermarked');

echo "<!DOCTYPE html><html lang='de'><head><meta charset='UTF-8'>
<title>Wasserzeichen-Verarbeitung</title>
<link href='https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css' rel='stylesheet'>
<style>
body{background:#0f172a;color:#e2e8f0;text-align:center;padding:40px;font-family:sans-serif;}
.card{background:#1e293b;border:none;padding:30px;display:inline-block;max-width:700px;width:100%;text-align:left;}
pre{background:#0f172a;color:#a1a1aa;padding:10px;border-radius:8px;overflow:auto;}
.btn-back{margin-top:25px;}
</style></head><body>";

echo "<h2>⚙️ Wasserzeichen-Verarbeitung</h2>";

if (!is_dir($source)) {
  echo "<p class='text-danger'>❌ Quellordner nicht gefunden: <code>$source</code></p>";
  echo "<a href='index.php' class='btn btn-outline-light btn-back'>⬅️ Zurück</a></body></html>";
  exit;
}
if (!is_dir($output)) @mkdir($output, 0777, true);

// --- Bilder suchen ---
$images = list_images($source);
$count = count($images);

if ($count === 0) {
  echo "<p class='text-warning'>Keine Bilder im Quellordner gefunden.</p>";
  echo "<a href='index.php' class='btn btn-outline-light btn-back'>⬅️ Zurück</a></body></html>";
  exit;
}

echo "<div class='card'><strong>Quelle:</strong> $source<br><strong>Ziel:</strong> $output<br><br>";
echo "<p>Starte Verarbeitung von <strong>$count</strong> Bildern...</p><pre>";

$processed = 0;

// --- Schleife durch alle Bilder ---
foreach ($images as $img) {
  $name = basename($img);
  $src  = $img;
  $dst  = $output . '/' . $name;

  // Preview-Skript zur Wasserzeichenerstellung verwenden
  $cmd = $_SERVER['DOCUMENT_ROOT'] . dirname($_SERVER['PHP_SELF']) . '/preview.php';
  $cmd = realpath($cmd);

  if (!is_file($src)) continue;

  [$srcW, $srcH, $srcType] = @getimagesize($src);
  if (!$srcW || !$srcH) continue;

  // Bild auslesen
  switch ($srcType) {
    case IMAGETYPE_PNG:  $srcImg = imagecreatefrompng($src); break;
    case IMAGETYPE_JPEG: $srcImg = imagecreatefromjpeg($src); break;
    case IMAGETYPE_WEBP: $srcImg = imagecreatefromwebp($src); break;
    case IMAGETYPE_GIF:  $srcImg = imagecreatefromgif($src); break;
    default: $srcImg = null;
  }

  if (!$srcImg) continue;

  // ---- Wasserzeichen anwenden ----
  $type     = $cfg['watermark_type'] ?? 'text';
  $opacity  = (int)($cfg['opacity'] ?? 60);
  $margin   = (int)($cfg['margin'] ?? 20);
  $pos      = $cfg['position'] ?? 'bottom-right';

  if ($type === 'image') {
    $wmRel  = $cfg['watermark_image'] ?? '';
    $wmPath = __DIR__ . '/' . ltrim($wmRel, '/');
    if (is_file($wmPath)) {
      [$wW, $wH, $wT] = getimagesize($wmPath);
      switch ($wT) {
        case IMAGETYPE_PNG:  $wm = imagecreatefrompng($wmPath); break;
        case IMAGETYPE_JPEG: $wm = imagecreatefromjpeg($wmPath); break;
        case IMAGETYPE_WEBP: $wm = imagecreatefromwebp($wmPath); break;
        case IMAGETYPE_GIF:  $wm = imagecreatefromgif($wmPath); break;
        default: $wm = null;
      }

      if ($wm) {
        imagealphablending($wm, true);
        imagesavealpha($wm, true);
        $scale = max(1, min(200, (int)($cfg['image_scale'] ?? 40)));
        $tW = (int)($srcW * ($scale / 100));
        $tH = (int)($wH * ($tW / $wW));
        $res = imagecreatetruecolor($tW, $tH);
        imagealphablending($res, false);
        imagesavealpha($res, true);
        imagecopyresampled($res, $wm, 0, 0, 0, 0, $tW, $tH, $wW, $wH);
        imagedestroy($wm);

        switch ($pos) {
          case 'top-left':     $x = $margin; $y = $margin; break;
          case 'top-right':    $x = $srcW - $tW - $margin; $y = $margin; break;
          case 'bottom-left':  $x = $margin; $y = $srcH - $tH - $margin; break;
          case 'center':       $x = ($srcW - $tW) / 2; $y = ($srcH - $tH) / 2; break;
          default:             $x = $srcW - $tW - $margin; $y = $srcH - $tH - $margin;
        }

        imagecopymerge($srcImg, $res, (int)$x, (int)$y, 0, 0, $tW, $tH, $opacity);
        imagedestroy($res);
      }
    }
  } else {
    $text  = $cfg['watermark_text'] ?? '© Volker';
    $font  = __DIR__ . '/' . ltrim($cfg['font_file'] ?? 'assets/DejaVuSans.ttf', '/');
    $fsize = (int)($cfg['font_size'] ?? 20);
    $colorHex = $cfg['color'] ?? '#FFFFFF';
    [$r, $g, $b] = sscanf($colorHex, "#%02x%02x%02x");
    $alpha = (127 - (127 * $opacity / 100));
    $color = imagecolorallocatealpha($srcImg, $r, $g, $b, $alpha);
    $box = imagettfbbox($fsize, 0, $font, $text);
    $tw = abs($box[2] - $box[0]);
    $th = abs($box[5] - $box[3]);
    switch ($pos) {
      case 'top-left':     $x = $margin; $y = $margin + $th; break;
      case 'top-right':    $x = $srcW - $tw - $margin; $y = $margin + $th; break;
      case 'bottom-left':  $x = $margin; $y = $srcH - $margin; break;
      case 'center':       $x = ($srcW - $tw) / 2; $y = ($srcH + $th) / 2; break;
      default:             $x = $srcW - $tw - $margin; $y = $srcH - $margin;
    }
    imagettftext($srcImg, $fsize, 0, (int)$x, (int)$y, $color, $font, $text);
  }

  imagejpeg($srcImg, $dst, 90);
  imagedestroy($srcImg);
  echo "✅ $name verarbeitet → $dst\n";
  $processed++;
}

echo "</pre><p><strong>Fertig!</strong> $processed von $count Bildern verarbeitet.</p>";
echo "<a href='index.php' class='btn btn-outline-light btn-back'>⬅️ Zurück</a>";
echo "</div></body></html>";
?>
